# 异步计算
:label:`sec_async`

今天的计算机是高度并行的系统，由多个CPU核、多个GPU、多个处理单元组成。通常每个CPU核有多个线程，每个设备通常有多个GPU，每个GPU有多个处理单元。总之，我们可以同时处理许多不同的事情，并且通常是在不同的设备上。不幸的是，Python并不善于编写并行和异步代码，至少在没有额外帮助的情况下不是好选择。归根结底，Python是单线程的，将来也是不太可能改变的。因此在诸多的深度学习框架中，MXNet和TensorFlow之类则采用了一种*异步编程*（asynchronous programming）模型来提高性能，而飞桨则使用了Python自己的调度器来实现不同的性能权衡。对于飞桨来说GPU操作在默认情况下是异步的。当你调用一个使用GPU的函数时，操作会排队到特定的设备上，但不一定要等到以后才执行。这允许我们并行执行更多的计算，包括在CPU或其他GPU上的操作。

因此，了解异步编程是如何工作的，通过主动地减少计算需求和相互依赖，有助于我们开发更高效的程序。这使我们能够减少内存开销并提高处理器利用率。



```python
import os
import subprocess
import numpy
import paddle
from paddle import nn
from d2l import paddle as d2l
```

## 通过后端异步处理


作为热身，考虑一个简单问题：我们要生成一个随机矩阵并将其相乘。让我们在NumPy和飞桨张量中都这样做，看看它们的区别。请注意，飞桨的`tensor`是在GPU上定义的。



```python
# GPU计算热身
a = paddle.randn(shape=(1000, 1000))
b = paddle.mm(a, a)

with d2l.Benchmark('numpy'):
    for _ in range(10):
        a = numpy.random.normal(size=(1000, 1000))
        b = numpy.dot(a, a)

with d2l.Benchmark('飞桨'):
    for _ in range(10):
        a = paddle.randn(shape=(1000, 1000))
        b = paddle.mm(a, a)
```

通过与基准输出比较，飞桨快了几个数量级。NumPy点积是在CPU上执行的，而飞桨矩阵乘法是在GPU上执行的，后者的速度要快得多。但巨大的时间差距表明一定还有其他原因。默认情况下，GPU操作在飞桨中是异步的。强制飞桨在返回之前完成所有计算，这种强制说明了之前发生的情况：计算是由后端执行，而前端将控制权返回给了Python。



```python
with d2l.Benchmark():
    for _ in range(10):
        a = paddle.randn(shape=(1000, 1000))
        b = paddle.mm(a, a)
    paddle.device.cuda.synchronize()
```

广义上说，飞桨有一个用于与用户直接交互的前端（例如通过Python），还有一个由系统用来执行计算的后端。如 :numref:`fig_frontends`所示，用户可以用各种前端语言编写Python程序，如Python和C++。不管使用的前端编程语言是什么，飞桨程序的执行主要发生在C++实现的后端。由前端语言发出的操作被传递到后端执行。后端管理自己的线程，这些线程不断收集和执行排队的任务。请注意，要使其工作，后端必须能够跟踪计算图中各个步骤之间的依赖关系。因此，不可能并行化相互依赖的操作。


![编程语言前端和深度学习框架后端](../img/frontends.png)
:width:`300px`
:label:`fig_frontends`

让我们看另一个简单例子，以便更好地理解依赖关系图。



```python
x = paddle.ones((1, 2))
y = paddle.ones((1, 2))
z = x * y + 2
z
```

![后端跟踪计算图中各个步骤之间的依赖关系](../img/asyncgraph.svg)
:label:`fig_asyncgraph`

上面的代码片段在 :numref:`fig_asyncgraph`中进行了说明。每当Python前端线程执行前三条语句中的一条语句时，它只是将任务返回到后端队列。当最后一个语句的结果需要被打印出来时，Python前端线程将等待C++后端线程完成变量`z`的结果计算。这种设计的一个好处是Python前端线程不需要执行实际的计算。因此，不管Python的性能如何，对程序的整体性能几乎没有影响。 :numref:`fig_threading`演示了前端和后端如何交互。

![前端和后端的交互](../img/threading.svg)
:label:`fig_threading`

## 障碍器与阻塞器


强制Python等待完成：

* 使用命令`paddle.device.cuda.synchronize()`等待该GPU设备上的所有计算完成。除非绝对必要，否则在实践中使用此运算符不是个好主意，因为它可能会导致较差的性能。
* 如果只想等待一个特定的变量可用，在飞桨中一般不需要考虑这个问题，因为飞桨系统内部会自动保证数据的同步和可用，我们只需要调用该变量即可。在单机多卡和多机多卡并行处理的时候，等待和确认所有计算完成是非常重要的问题。

让我们看看这在实践中是如何运作的。



```python
with d2l.Benchmark('等待计算完成'):
    b = paddle.dot(a, a)
    paddle.device.cuda.synchronize()

with d2l.Benchmark('不等待计算完成'):
    b = paddle.dot(a, a)


```

两个操作的完成时间还是能看出明显差别的。除了显式地阻塞操作之外，我们还建议要注意*隐式*的阻塞器。打印变量就是一个阻塞器，因为其要求变量可用。最后，通过`z.numpy()`转换为NumPy类型的变量和通过`z.item()`转换为标量也是阻塞器。

频繁地将少量数据从飞桨的作用域复制到NumPy，可能会破坏原本高效代码的性能，因为每一个这样的操作都需要使用计算图来求得所有的中间结果，从而获得相关项，然后才能做其它事情。



```python
with d2l.Benchmark('numpy conversion'):
    b = paddle.dot(a, a)
    b.numpy()

with d2l.Benchmark('scalar conversion'):
    b = paddle.dot(a, a)
    b.sum().item()
```

## 改进计算


在重度多线程的系统中（即使普通笔记本电脑也有4个或更多线程，然而在多插槽服务器上这个数字可能超过256），调度操作的开销可能会变得非常大。这也是极度希望计算和调度是异步和并行的原因。为了说明这样做的好处，让我们看看按顺序（同步执行）或异步执行多次将变量递增$1$会发生什么情况。我们通过在每个加法之间插入`paddle.device.cuda.synchronize()`障碍器来同步执行。



```python
with d2l.Benchmark('同步执行'):
    for _ in range(10000):
        y = x + 1
        paddle.device.cuda.synchronize()

with d2l.Benchmark('异步执行'):
    for _ in range(10000):
        y = x + 1
    paddle.device.cuda.synchronize()
```

Python前端线程和C++后端线程之间的简化交互可以概括如下：

1. 前端命令后端将计算任务`y = x + 1`插入队列。
1. 然后后端从队列接收计算任务并执行。
1. 然后后端将计算结果返回到前端。
假设这三个阶段的持续时间分别为$t_1, t_2, t_3$。如果不使用异步编程，执行10000次计算所需的总时间约为$10000 (t_1+ t_2 + t_3)$。如果使用异步编程，因为前端不必等待后端为每个循环返回计算结果，执行$10000$次计算所花费的总时间可以减少到$t_1 + 10000 t_2 + t_3$（假设$10000 t_2 > 9999t_1$）。


## 小结

* 深度学习框架可以将Python前端的控制与后端的执行解耦，使得命令可以快速地异步插入后端、并行执行。
* 异步产生了一个相当灵活的前端，但请注意：过度填充任务队列可能会导致内存消耗过多。建议对每个小批量进行同步，以保持前端和后端大致同步。
* 芯片供应商提供了复杂的性能分析工具，以获得对深度学习效率更精确的洞察。


## 练习


1. 在CPU上，对本节中相同的矩阵乘法操作进行基准测试。你仍然可以通过后端观察异步吗？


[Discussions](https://discuss.d2l.ai/t/2791)

